import { supabase } from '@/config/supabase';
import {
  Withdrawal,
  WalletSummary,
  BankAccount,
  UpdateBankAccountRequest,
  UpdateWithdrawalStatusRequest,
  UpdateGlobalWithdrawalStatusRequest
} from './types';

export const withdrawalsService = {
  // Get withdrawal history and wallet summary
  async getWithdrawalHistoryAndSummary(vendor_id: string) {
    // 1. Get withdrawals
    const { data: withdrawals, error } = await supabase.from('withdrawals').select('*').eq('vendor_id', vendor_id).order('created_at', { ascending: false });
    // 2. Get wallet summary (simulate: sum available and pending earnings)
    const { data: summary } = await supabase.rpc('get_wallet_summary', { vendor_id });
    return { data: { withdrawals, wallet: summary }, error };
  },

  // Update bank account (max once per 30 days)
  async updateBankAccount(vendor_id: string, req: UpdateBankAccountRequest) {
    // Check last update
    const { data: vendor } = await supabase.from('vendors').select('last_updated_bank_account').eq('id', vendor_id).single();
    if (vendor?.last_updated_bank_account) {
      const last = new Date(vendor.last_updated_bank_account);
      const now = new Date();
      const diffDays = (now.getTime() - last.getTime()) / (1000 * 60 * 60 * 24);
      if (diffDays < 30) {
        return { error: { message: 'Bank account can only be updated once every 30 days.' } };
      }
    }
    const { data, error } = await supabase.from('vendors').update({
      ...req,
      last_updated_bank_account: new Date().toISOString()
    }).eq('id', vendor_id).select().single();
    return { data, error };
  },

  // Admin: pause/resume withdrawals for vendor
  async updateVendorWithdrawalStatus(vendor_id: string, req: UpdateWithdrawalStatusRequest) {
    const { data, error } = await supabase.from('vendors').update({ withdrawals_paused: req.withdrawals_paused }).eq('id', vendor_id).select().single();
    return { data, error };
  },

  // Admin: pause/resume withdrawals globally
  async updateGlobalWithdrawalStatus(req: UpdateGlobalWithdrawalStatusRequest) {
    const { data, error } = await supabase.from('admin_settings').update({ global_withdrawals_paused: req.global_withdrawals_paused }).select().single();
    return { data, error };
  },

  // Admin: manual trigger of withdrawal job (simulate)
  async triggerWithdrawalJob() {
    // In real app, enqueue job or run CRON logic
    // Simulate: return success
    return { data: { triggered: true } };
  }
}; 